  ' -----------------------------------------------------------
  ' Program for Micromite to communicate with the Elecrow GY-68
  ' Barometer module, based on the Bosch BMP180 chip.
  ' Written by Jim Rowe for Silicon Chip.
  ' Last revision 25/7/2017 at 1:35 pm
  '
  ' Notes:
  ' 1. Communication with the BMP180 chip is via the Micromite's
  ' I2C port, which uses pin 17 for SCL and pin 18 for SDA.
  ' 2. Although the chip's I2C address is specified as EFh for
  ' reading and EEh for writing, MMBasic's I2C function expects a
  ' 7-bit I2C address since it adds the lsb read/write bit automatically.
  ' So in this program we use 77h as the chip's I2C address.
  ' ----------------------------------------------------------
  
  OPTION AUTORUN ON
  OPTION EXPLICIT
  
  DIM AS INTEGER AC1 = 0          ' BMP-180 cal parameter AC1 (16 bits)
  DIM AS INTEGER AC2 = 0          ' BMP-180 cal parameter AC2 (16 bits)
  DIM AS INTEGER AC3 = 0          ' BMP-180 cal parameter AC3 (16 bits)
  DIM AS INTEGER AC4 = 0          ' BMP-180 cal parameter AC4 (16 bits)
  DIM AS INTEGER AC5 = 0          ' BMP-180 cal parameter AC5 (16 bits)
  DIM AS INTEGER AC6 = 0          ' BMP-180 cal parameter AC6 (16 bits)
  DIM AS INTEGER B1 = 0           ' BMP-180 cal parameter B1 (16 bits)
  DIM AS INTEGER B2 = 0           ' BMP-180 cal parameter B2 (16 bits)
  DIM AS INTEGER MB = 0           ' BMP-180 cal parameter MB (16 bits)
  DIM AS INTEGER MC = 0           ' BMP-180 cal parameter MC (16 bits)
  DIM AS INTEGER MD = 0           ' BMP-180 cal parameter MD (16 bits)
  DIM AS INTEGER UT = 0           ' uncompensated temperature reading
  DIM AS INTEGER UP = 0           ' uncompensated pressure reading

  DIM AS FLOAT Temp               ' compensated temperature reading
  DIM AS FLOAT Pres               ' compensated pressure reading
  DIM AS FLOAT RelP               ' pressure at MSL taking altitude into ac.
  DIM AS FLOAT X1, X2, X3, B3, B4, B5, B6, B7   ' working variables
  
  DIM AS INTEGER Alt = 50         ' nominal altitude above sea level (m)
                                  ' (Change to suit your location)
  Const DBlue = RGB(0,0,128)
  CONST Bone = RGB(255,255,192)
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  
  PIN(17) = 0             ' set pin 17 to low and
  SETPIN 17, DOUT         ' declare it a digital output (SCL)

  DIM INTEGER CData(21)   ' one-dim array to save cal data bytes from EEPROM
  DIM INTEGER TByte(1)    ' one dim array to save raw UT data bytes
  DIM INTEGER PByte(2)    ' one-dim array to save raw UP data bytes
  
  I2C OPEN 100, 200       ' enable the Micromite I2C module in master mode
  ' ----------------------------------------------------------------------
  ' first show the opening screen
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), DBlue
  TEXT MM.HRes/2, MM.VRes*3/32, "SILICON CHIP", CM, 1, 3, Red, DBlue
  TEXT MM.HRes/2, MM.VRes*7/32, "BMP180 Module Checkout Program", CM, 1, 1, White, DBlue
  TEXT MM.HRes/2, MM.VRes*5/16, "**                    **", CM, 1, 1, Green, DBlue
  RBOX 4, MM.VRes*3/8, MM.HRes-10, MM.VRes*5/8-6, 8, Black, Bone
  TEXT MM.HRes/2, MM.VRes*7/16, "Temperature:", CM, 1,2, Black, Bone
  TEXT MM.HRes/2, MM.VRes*9/16, "00.00 deg C", CM, 1,2, Red, Bone
  TEXT MM.HRes/2, MM.VRes*21/32, "Absolute Pressure: 0000.00 mBar", CM, 1,1, Black, Bone
  TEXT MM.HRes/2, MM.VRes*25/32, "Pressure RelMSL:", CM, 1,2, Black, Bone
  TEXT MM.HRes/2, MM.VRes*29/32, "0000.00 hPa", CM, 1,2, Red, Bone
  
  ' now go fetch the BMP180 chip's calibration data
    I2C WRITE &H77,0,1,&HAA     ' send first EEPROM reg address to be read
    I2C READ &H77,0,22,CData()  ' then read 22 bytes & save in CData() array

  ' The cal parameter bytes should now have been retrieved from the EEPROM
  ' registers of the BMP180, but we now need to assemble each pair and then
  ' mask them back to 16 bits. For all except AC4, AC5 and AC6 we also need
  ' to convert them into signed integers if they are >= 32768
  AC1 = ((CData(0) << 8) OR CData(1)) AND &H000000000000FFFF
  IF AC1 >= 32768 THEN AC1 = (AC1 - 65535)
  AC2 = ((CData(2) << 8) OR CData(3)) AND &H000000000000FFFF
  IF AC2 >= 32768 THEN AC2 = (AC2 - 65535)
  AC3 = ((CData(4) << 8) OR CData(5)) AND &H000000000000FFFF
  IF AC3 >= 32768 THEN AC3 = (AC3 - 65535)
  AC4 = ((CData(6) << 8) OR CData(7)) AND &H000000000000FFFF
  AC5 = ((CData(8) << 8) OR CData(9)) AND &H000000000000FFFF
  AC6 = ((CData(10) << 8) OR CData(11)) AND &H000000000000FFFF
  B1 = ((CData(12) << 8) OR CData(13)) AND &H000000000000FFFF
  IF B1 >= 32768 THEN B1 = (B1 - 65535)
  B2 = ((CData(14) << 8) OR CData(15)) AND &H000000000000FFFF
  IF B2 >= 32768 THEN B2 = (B2 - 65535)
  MB = ((CData(16) << 8) OR CData(17)) AND &H000000000000FFFF
  IF MB >= 32768 THEN MB = (MB - 65535)
  MC = ((CData(18) << 8) OR CData(19)) AND &H000000000000FFFF
  IF MC >= 32768 THEN MC = (MC - 65535)
  MD = ((CData(20) << 8) OR CData(21)) AND &H000000000000FFFF
  IF MD >= 32768 THEN MD = (MD - 65535)
  ' now that's done, indicate on screen
  TEXT MM.HRes/2, MM.VRes*5/16, "** BMP180 initialised **", CM, 1, 1, Green, DBlue
  PRINT "BMP180 Initialised"      ' also print on console

  ' main program loop starts here
  DO
    GoReadTemp          ' now get an uncompensated temperature reading
    GoReadPressure      ' and also the uncompensated pressure
    DoTheMaths          ' then do the maths and
    DispRes             ' display the result
    PAUSE 1000          ' set to loop about once per second
  LOOP
END ' end of main part of program, subroutines follow
  
  ' *****************************************************************
  ' subroutine to read uncompensated temperature (UT) from BMP180
SUB GoReadTemp
  I2C WRITE &H77,0,2,&HF4,&H2E  ' send 2Eh to control register F4
  PAUSE 5                       ' then wait 5ms for measurement
  I2C WRITE &H77,0,1,&HF6       ' send first address to read (F6h)
  I2C READ &H77,0,2, TByte()    ' then read the two UT bytes
  ' then glue them together to get the UT reading
  UT = ((TByte(0) << 8) OR TByte(1)) AND &H000000000000FFFF
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to read uncompensated pressure (UP) from BMP180
SUB GoReadPressure
  I2C WRITE &H77,0,2,&HF4,&H34  ' send 34h to control register F4
  PAUSE 10                      ' then wait 10ms for measurement
  I2C WRITE &H77,0,1, &HF6      ' send first address to read (F6h)
  I2C READ &H77,0,3, PByte()    ' then read the three UP bytes
  ' then glue them together to get the UP reading (oss = 00)
  UP = (((PByte(0) << 16) OR (PByte(1) << 8) OR PByte(2)) >> 8) AND &H0000000000FFFFFF
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to 'do the maths' for calculating compensated Temp and Pres
  ' based on expressions given in Bosch's BMP180 datasheet on page 15.
SUB DoTheMaths
  X1 = ((UT - AC6) * AC5)/32768
  X2 = (MC * 2048)/(X1 + MD)
  B5 = X1 + X2
  Temp = (B5 + 8)/16        ' this should be the comp temp in 0.1 deg C
  Temp = (Temp/10)          ' so now it should be in deg C.
  '
  ' now to crunch the data to get the pressure (Pres)
  B6 = B5 - 4000
  X1 = (B2 * (B6 * B6/4096))/2048
  X2 = (AC2 * B6)/2048
  X3 = X1 + X2
  B3 = (((AC1 * 4) + X3) + 2)/4
  X1 = (AC3*B6)/8192
  X2 = (B1 * (B6 * B6/4096))/65536
  X3 = ((X1 + X2) + 2)/4
  B4 = AC4 * (X3 + 32768)/32768
  B7 = (UP - B3) * 50000
  IF (B7 < &H80000000) THEN
    Pres = (B7 * 2)/B4
  ELSE
    Pres = (B7/B4) * 2
  END IF
  X1 = (Pres/256) * (Pres/256)
  X1 = (X1 * 3038)/65536
  X2 = (-7357*Pres)/65536
  Pres = Pres + (X1 + X2 + 3791)/16     ' this gives pressure in Pascals,
  Pres = (Pres * 0.01)                  ' so convert to mBar (= hPa)
  RelP = Pres/((1 - (Alt/44330))^5.255)  ' also work out RelP at MSL
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to display the calculated temp and pressure
SUB DispRes
  TEXT MM.HRes/2, MM.VRes*9/16, STR$(Temp,2,2) + " deg C", CM, 1,2, Red, Bone
  TEXT MM.HRes/2, MM.VRes*21/32, "Absolute Pressure: " + STR$(Pres,4,2) + " mBar",CM,1,1,Black,Bone
  TEXT MM.HRes/2, MM.VRes*29/32, STR$(RelP,4,2) + " hPa", CM, 1,2, Red, Bone
  PRINT "Temp = ", STR$(Temp,2,2), " degC, Pressure = ", STR$(Pres,4,2), " mBar, MSLP = ", STR$(RelP,4,2), " hPa"
  END SUB
  ' -------------------------------------------------------------------
